;; base.lisp

(use-package :cl-ppcre)
(use-package :on-lisp)
(use-package :haskellish)

(defparameter epsilon 1d-30) ; arbitrarily small value 
(defparameter *sigfigs* 6)

(defun tr-d-to-e (str)
  (regex-replace-all "d" str "e"))

(defun plot (fn start end steps &optional (inclusive t))
  (let* ((step-size (/ (- end start) steps))
         (start* (if inclusive start (+ start step-size)))
         (end* (if inclusive end (- end step-size))))
    (loop for x from start* to end* by step-size
       collect (list x (funcall fn x)))))

(defun write-data (data filename)
  (with-open-file (stream filename :direction :output :if-exists :supersede)
    (loop for inner-list in data
         do (let ((row (format nil "~{~e ~}~%" inner-list)))
              (format stream (tr-d-to-e row))))))

(defun plot-data (fn start end step-size filename)
  (write-data (plot fn start end step-size) filename))

;; I was getting some stack overflow problems.  So I found this
;; trampoline method to get around it.
;;
;; http://bc.tech.coop/blog/040613.html
(defun run-trampolined (f &rest args)
  (catch 'done
    (loop
       (let ((new-f-and-args (multiple-value-list (apply f args))))
         (setf f (first new-f-and-args)
               args (rest new-f-and-args))))))

(defun sigfigs (x dx)
  (let ((l (log x 10))
        (dl (log dx 10)))
     (abs (- (truncate dl) (truncate l)))))

(defun sigfigs-range (x1 x2)
  (apply #'sigfigs (median-delta x1 x2)))

(defun scale-plot (plot-list &rest scale-factors)
  (loop for list in plot-list
       collect (mapcar #'* list scale-factors)))

(defun latex-number (n &optional (sigfigs *sigfigs*))
  (register-groups-bind (num exponent)
      ("([\\d\\.]+)d([\\d\\+\\-]+)" (format nil (format nil "~~,~ae" (1- sigfigs)) (coerce n 'double-float)))
    (format nil "$ ~a \\cdot 10^{~a}$" num exponent)))

(defun split-table (root-list &optional (chunk 35))
  (and root-list 
       (cons (take root-list chunk) 
             (split-table (drop root-list chunk) chunk))))
    
(defun write-table (list filename)
  (with-open-file (stream filename :direction :output :if-exists :supersede)
    (loop for inner-list in list
         do (format stream "~{~a ~^&~} \\\\~%" inner-list))))

(defun average (xs)
  (/ (reduce #'+ xs) (length xs)))

(defun standard-deviation (xs)
  (let* ((xbar (average xs))
         (n (length xs))
         (variance (/ (loop for x in xs
                         sum (expt (- x xbar) 2)) (- n 1))))
    (sqrt variance)))

(defun find-min-max (xs)
  (loop for x in xs
       maximizing x into max
       minimizing x into min
       finally (return (list min max))))



